<?php

if (!defined('ABSPATH'))
    exit;

class WPGens_RAF_Api
{

    /**
     * @var    object
     * @access  private
     * @since    1.0.0
     */
    private static $_instance = null;

    /**
     * The version number.
     * @var     string
     * @access  public
     * @since   1.0.0
     */
    public $_version;

    /**
     * WPGens Events Listing
     * @var     string
     * @access  public
     * @since   1.0.0
     */
    private $events;

    /**
     * WPGens Orders History
     * @var     string
     * @access  public
     * @since   1.0.0
     */
    private $orders_history;

    /**
     * WPGens Customers History
     * @var     string
     * @access  public
     * @since   1.0.0
     */
    private $customers_history;

    public function __construct(WPGENS_RAF_Events $events, WPGENS_RAF_Orders_History $orders_history, WPGENS_RAF_Customers_History $customers_history)
    {
        $this->events = $events;
        $this->orders_history = $orders_history;
        $this->customers_history = $customers_history;

        add_action('rest_api_init', function () {
            // Settings
            register_rest_route('gens-raf/v1', '/settings/', array(
                'methods' => 'GET',
                'callback' => array($this, 'get_settings'),
                'permission_callback' => array($this, 'get_permission')
            ));
            register_rest_route('gens-raf/v1', '/settings/', array(
                'methods' => 'POST',
                'callback' => array($this, 'set_settings'),
                'permission_callback' => array($this, 'get_permission')
            ));

            // Licence
            register_rest_route('gens-raf/v1', '/license/', array(
                'methods' => 'GET',
                'callback' => array($this, 'get_license'),
                'permission_callback' => array($this, 'get_permission')
            ));
            register_rest_route('gens-raf/v1', '/license/deactivate', array(
                'methods' => 'POST',
                'callback' => array($this, 'deactivate_license'),
                'permission_callback' => array($this, 'get_permission')
            ));
            register_rest_route('gens-raf/v1', '/license/', array(
                'methods' => 'POST',
                'callback' => array($this, 'set_license'),
                'permission_callback' => array($this, 'get_permission')
            ));

            // Events
            register_rest_route('gens-raf/v1', '/events/(?P<page>\d+)', array(
                'methods' => 'GET',
                'callback' => array($this, 'get_events'),
                'args' => array(
                    'page' => array(
                        'validate_callback' => function ($param, $request, $key) {
                            return is_numeric($param);
                        }
                    ),
                ),
                'permission_callback' => array($this, 'get_permission')
            ));

            // Orders
            register_rest_route('gens-raf/v1', '/orders/(?P<page>\d+)', array(
                'methods' => 'GET',
                'callback' => array($this, 'get_orders'),
                'args' => array(
                    'page' => array(
                        'validate_callback' => function ($param, $request, $key) {
                            return is_numeric($param);
                        }
                    ),
                ),
                'permission_callback' => array($this, 'get_permission')
            ));

            // Customers
            register_rest_route('gens-raf/v1', '/customers/(?P<days>\d+)', array(
                'methods' => 'GET',
                'callback' => array($this, 'get_customers'),
                'args' => array(
                    'days' => array(
                        'validate_callback' => function ($param, $request, $key) {
                            return is_numeric($param);
                        }
                    ),
                ),
                'permission_callback' => array($this, 'get_permission')
            ));
        });
    }

    /**
     *
     * Ensures only one instance of AWDP is loaded or can be loaded.
     *
     * @since 1.0.0
     * @static
     * @see WordPress_Plugin_Template()
     * @return Main AWDP instance
     */
    public static function instance(WPGENS_RAF_Events $events, WPGENS_RAF_Orders_History $orders_history, WPGENS_RAF_Customers_History $customers_history)
    {
        if (is_null(self::$_instance)) {
            self::$_instance = new self($events, $orders_history, $customers_history);
        }
        return self::$_instance;
    }

    public function allowed_html_settings()
    {
        return array('gens_raf_share_text', 'gens_raf_myaccount_text', 'gens_raf_share_text', 'gens_raf_email_body', 'gens_raf_guest_text', 'gens_raf_email_message', 'gens_raf_buyer_email_message');
    }

    public function registered_settings()
    {

        $prefix = 'gens_raf_';
        $options = array(
            $prefix . 'disable',
            $prefix . 'cookie_time',
            $prefix . 'min_ref_order',
            $prefix . 'cookie_remove',
            $prefix . 'allow_guests',
            $prefix . 'hide_no_orders',
            $prefix . 'hide_no_orders_text',
            $prefix . 'referral_codes',
            $prefix . 'subscription',
            $prefix . 'subscription_all_coupons',
            $prefix . 'subscription_exclude_shipping',
            $prefix . 'subscription_partial_use',

            // Product
            $prefix . 'tabs_disable',
            $prefix . 'share_text',
            $prefix . 'guest_text',
            // My Account
            $prefix . 'my_account_url',
            $prefix . 'myaccount_text',
            // Share
            $prefix . 'twitter_via',
            $prefix . 'twitter_title',
            $prefix . 'whatsapp',
            $prefix . 'viber',
            $prefix . 'linkedin',
            $prefix . 'pinterest',
            // Emails
            $prefix . 'disable_emails',
            $prefix . 'use_woo_mail',
            $prefix . 'email_subject',
            $prefix . 'email_heading',
            $prefix . 'email_message',
            $prefix . 'buyer_email_subject',
            $prefix . 'buyer_email_message',
            $prefix . 'email_subject_share',
            $prefix . 'email_heading_share',
            $prefix . 'email_body',
            $prefix . 'email_from',
            $prefix . 'email_hide',
            // Fraud
            $prefix . 'fraud_disable_ip',
            $prefix . 'allow_existing',
            $prefix . 'fraud_disable_street',
            $prefix . 'fraud_email_address',
            // Coupon
            $prefix . 'generate_for_registration',
            $prefix . 'generate_for_order',
            $prefix . 'coupon_type',
            $prefix . 'coupon_amount',
            $prefix . 'product_ids',
            $prefix . 'product_exclude_ids',
            $prefix . 'product_categories',
            $prefix . 'exclude_product_categories',
            $prefix . 'coupon_duration',
            $prefix . 'min_order',
            $prefix . 'individual_use',
            $prefix . 'exclude_sale_items',
            $prefix . 'free_shipping',
            $prefix . 'limit_usage',
            $prefix . 'nth_coupon',

            // Friend Coupon
            $prefix . 'friend_coupon_type',
            $prefix . 'friend_coupon_amount',
            $prefix . 'friend_product_ids',
            $prefix . 'friend_product_exclude_ids',
            $prefix . 'friend_product_categories',
            $prefix . 'friend_exclude_product_categories',
            $prefix . 'friend_coupon_duration',
            $prefix . 'friend_min_order',
            $prefix . 'friend_individual_use',
            $prefix . 'friend_exclude_sale_items',
            $prefix . 'friend_free_shipping',
            $prefix . 'friend_limit_usage',
            $prefix . 'friend_enable',
            $prefix . 'guest_enable',
            $prefix . 'guest_coupon_code',
            $prefix . 'guest_coupon_message',

        );
        return $options;
    }

    function get_settings($data)
    {

        foreach ($this->registered_settings() as $key) {
            if ($value = get_option($key)) {
                // solution for wpgens refer a friend which had "yes","no" saved for boolean.
                switch ($value) {
                    case 'no':
                        $result[$key] = false;
                        break;
                    case 'yes':
                        $result[$key] = true;
                        break;
                    default:
                        $result[$key] = $value;
                        break;
                }
            }
        }

        return new WP_REST_Response($result, 200);
    }

    function set_settings($data)
    {
        $fields = $this->registered_settings();
        $allowed_html = $this->allowed_html_settings();

        $data = $data->get_params();

        foreach ($data as $key => $value) {
            if (in_array($key, $fields)) {
                if (false === get_option($key)) {
                    $sanitized_data = in_array($key, $allowed_html) ? wp_kses_post($data[$key]) : sanitize_text_field($data[$key]);
                    add_option($key, $sanitized_data);
                } else {
                    $sanitized_data = in_array($key, $allowed_html) ? wp_kses_post($data[$key]) : sanitize_text_field($data[$key]);
                    update_option($key, wp_kses_post($sanitized_data));
                }
            }
        }

        // $addition_settings                      = get_option('awdp_addition_settings') ? get_option('awdp_addition_settings') : [];
        // $result['use_regular']                  = array_key_exists ( 'use_regular', $addition_settings ) ? $addition_settings['use_regular'] : false;

        return new WP_REST_Response($data, 200);
    }


    function set_license($data)
    {
        $data = $data->get_params();
        if (!isset($data['gens_raf_license_key'])) {
            return new WP_REST_Response(array('message' => 'Something went wrong!'), 403);
        }

        $response = WPGens_RAF_Licence::gens_raf_activate_licence($data['gens_raf_license_key']);
        return new WP_REST_Response($response, 200);
    }

    function get_license()
    {
        $response = WPGens_RAF_Licence::gens_raf_check_licence();
        return new WP_REST_Response($response, 200);
    }

    function deactivate_license()
    {
        $response = WPGens_RAF_Licence::gens_raf_deactivate_licence();
        return new WP_REST_Response($response, 200);
    }

    function get_events(WP_REST_Request $request)
    {
        $params = $request->get_params();
        if (isset($params['page']) && is_numeric($params['page'])) {
            $response = $this->events->get_events(intval($params['page']));
            return new WP_REST_Response($response, 200);
        }
        return new WP_REST_Response(array('message' => 'Something went wrong!'), 403);
    }

    function get_orders(WP_REST_Request $request)
    {
        $params = $request->get_params();
        if (isset($params['page']) && is_numeric($params['page'])) {
            $response = $this->orders_history->get_orders(intval($params['page']));
            return new WP_REST_Response($response, 200);
        }
        return new WP_REST_Response(array('message' => 'Something went wrong!'), 403);
    }

    function get_customers(WP_REST_Request $request)
    {
        $params = $request->get_params();
        if (isset($params['days']) && is_numeric($params['days'])) {
            $days = sanitize_text_field($params['days']);
            $response = $this->customers_history->get_customers(intval($days));
            return new WP_REST_Response($response, 200);
        }
        return new WP_REST_Response(array('message' => 'Something went wrong!'), 403);
    }

    /**
     * Permission Callback
     **/
    public function get_permission()
    {
        if (current_user_can('administrator') || current_user_can('manage_woocommerce')) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Cloning is forbidden.
     *
     * @since 1.0.0
     */
    public function __clone()
    {
        _doing_it_wrong(__FUNCTION__, __('Cheatin&#8217; huh?'), $this->_version);
    }

    /**
     * Unserializing instances of this class is forbidden.
     *
     * @since 1.0.0
     */
    public function __wakeup()
    {
        _doing_it_wrong(__FUNCTION__, __('Cheatin&#8217; huh?'), $this->_version);
    }
}

$events = new WPGENS_RAF_Events();

$orders_history = new WPGENS_RAF_Orders_History();

$customers_history = new WPGENS_RAF_Customers_History();

WPGens_RAF_Api::instance($events, $orders_history, $customers_history);
