<?php
/**
 * Plugin Name: WooCommerce Amazon S3 Storage
 * Plugin URI: https://woocommerce.com/products/amazon-s3-storage/
 * Description: Store your downloadbable products on Amazon S3 offering faster downloads for your customers and more security for your product.
 * Version: 2.1.22
 * Author: WooCommerce
 * Author URI: https://woocommerce.com/
 * Requires at least: 3.8
 * Tested up to: 5.5
 * WC tested up to: 4.4
 * WC requires at least: 2.6
 * Text Domain: wc_amazon_s3
 * Woo: 18663:473bf6f221b865eff165c97881b473bb
 *
 * Copyright: © 2020 WooCommerce
 * License: GNU General Public License v3.0
 * License URI: https://www.gnu.org/licenses/gpl-3.0.html
 *
 * @package woocommerc-amazon-s3-storage
 */

// Plugin init hook.
add_action( 'plugins_loaded', 'wc_amazon_s3_init' );

/**
 * Initialize plugin.
 */
function wc_amazon_s3_init() {

	if ( ! class_exists( 'WooCommerce' ) ) {
		add_action( 'admin_notices', 'wc_amazon_s3_woocommerce_deactivated' );
		return;
	}

	define( 'WC_AMAZON_S3_STORAGE_VERSION', '2.1.22' ); // WRCS: DEFINED_VERSION.

	/**
	 * Localisation
	 **/
	load_plugin_textdomain( 'wc_amazon_s3', false, dirname( plugin_basename( __FILE__ ) ) . '/lang' );

	/**
	 * WooCommerce_Amazon_S3_Storage class
	 **/
	if ( ! class_exists( 'WooCommerce_Amazon_S3_Storage' ) ) {

		class WooCommerce_Amazon_S3_Storage {

			/**
			 * Class Variables
			 **/
			public $settings_name = 'woo_amazon_s3_storage';
			public $credentials = array();
			var $disable_ssl;

			/**
			 * Constructor
			 **/
			function __construct() {
				// Load admin settings.
				$admin_settings = get_option( $this->settings_name );
				$this->credentials['key'] = $admin_settings['amazon_access_key'];
				$this->credentials['secret'] = $admin_settings['amazon_access_secret'];
				$this->disable_ssl = ( ! empty( $admin_settings['amazon_disable_ssl'] ) ? $admin_settings['amazon_disable_ssl'] : 0 );

				// Create Menu under WooCommerce Menu
				add_action( 'admin_menu', array( $this, 'register_menu' ) );
				add_filter( 'woocommerce_downloadable_product_name', array( $this, 'wc2_product_download_name' ), 10, 4 );
				add_filter( 'woocommerce_file_download_path', array( $this, 'wc2_product_download' ), 1, 3 );
				register_activation_hook( __FILE__, array( $this, 'upgrade' ) );

				// Add amazon_s3 shortcode
				add_shortcode( 'amazon_s3', array( $this, 'amazon_shortcode' ) );

				/**
				 * Added since 2.1.5
				 * @todo remove two minor versions later.
				 */
				add_action( 'admin_init', array( $this, 'temp_upgrade' ) );

				add_action( 'plugin_action_links_' . plugin_basename( __FILE__ ), array( $this, 'plugin_action_links' ) );

				add_action( 'init', array( $this, 'includes' ) );
			}

			public function includes() {
				require_once( dirname( __FILE__ ) . '/woocommerce-amazon-s3-storage-privacy.php' );
			}

			public function temp_upgrade() {
				$version = get_option( 'wc_amazon_s3_storage_version', '' );

				// In 2.1.5 we changed transient to be hourly. So we must clear the current
				// transients first as those do not expire which causes region to be stale.
				if ( empty( $version ) || version_compare( $version, '2.1.5', '<' ) ) {
					global $wpdb;

					$wpdb->query( "DELETE FROM $wpdb->options WHERE option_name LIKE '%s3-region-%'" );
					update_option( 'wc_amazon_s3_storage_version', WC_AMAZON_S3_STORAGE_VERSION );
				}
			}

			/**
			 * Add custom plugin action links.
			 *
			 * @since 2.1.8
			 *
			 * @param array $links Links.
			 *
			 * @return array Links.
			 */
			public function plugin_action_links( $links ) {
				$plugin_links = array(
					'<a href="' . admin_url( 'admin.php?page=woo_amazon_s3_storage' ) . '">' . __( 'Settings', 'wc_amazon_s3' ) . '</a>',
					'<a href="https://docs.woocommerce.com/">' . __( 'Support', 'wc_amazon_s3' ) . '</a>',
					'<a href="https://docs.woocommerce.com/document/amazon-s3-storage">' . __( 'Docs', 'wc_amazon_s3' ) . '</a>',
				);

				return array_merge( $plugin_links, $links );
			}

			function register_menu() {
				add_submenu_page( 'woocommerce', __( 'WooCommerce Amazon S3 Storage', 'wc_amazon_s3' ), __( 'Amazon S3 Storage', 'wc_amazon_s3' ), 'manage_woocommerce', 'woo_amazon_s3_storage', array( &$this, 'menu_setup' ) );
			}

			function upgrade() {
				$args = array(
					'post_type' => array( 'product', 'product_variation' ),
					'meta_key' => '_amazon_s3_bucket',
					'posts_per_page' => -1,
				);
				$upgrade_query = new WP_Query( $args );
				while ( $upgrade_query->have_posts() ) {
					$upgrade_query->the_post();
					global $post;
					$use_s3 = get_post_meta( $post->ID, '_use_amazon_s3', true );
					if ( 'yes' === $use_s3 ) {
						$bucket = get_post_meta( $post->ID, '_amazon_s3_bucket', true );
						$object = get_post_meta( $post->ID, '_amazon_s3_object', true );
						$shortcode = '[amazon_s3 bucket=' . $bucket . ' object=' . $object . ']';
						$file_paths = get_post_meta( $post->ID, '_file_paths', true );
						$old_file_paths = $file_paths;
						if ( is_array( $file_paths ) || empty( $file_paths ) ) {
							if ( empty( $file_paths ) ) {
								$file_paths = array( $shortcode );
								update_post_meta( $post->ID, '_file_paths', $file_paths, $old_file_paths );
								continue;
							}
							if ( ! in_array( $shortcode, $file_paths ) ) {
								$file_paths[] = $shortcode;
								update_post_meta( $post->ID, '_file_paths', $file_paths, $old_file_paths );
								//delete_post_meta( $post->ID, '_use_amazon_s3' );
								//delete_post_meta( $post->ID, '_amazon_s3_bucket' );
								//delete_post_meta( $post->ID, '_amazon_s3_object' );
							}
						}
					}
				}

				update_option( 'wc_amazon_s3_storage_version', WC_AMAZON_S3_STORAGE_VERSION );
			}

			function array_to_options( $array, $selected, $escape = '' ) {
				$options = '';
				foreach ( $array as $id => $value ) {
					$options .= '<option value="' . esc_attr( $id ) . '" ' . selected( $array, $selected ) . '>' . esc_attr( $value ) . '</option>';
				}
				return $options . $escape;
			}

			function menu_setup() {
				$admin_options = get_option( $this->settings_name );

				// Save values if submitted
				if ( isset( $_POST['woo_amazon_access_key'] ) ) {
					$admin_options['amazon_access_key'] = $_POST['woo_amazon_access_key'];
				}
				if ( isset( $_POST['woo_amazon_access_secret'] ) ) {
					$admin_options['amazon_access_secret'] = $_POST['woo_amazon_access_secret'];
				}
				if ( isset( $_POST['woo_amazon_url_period'] ) ) {
					$admin_options['amazon_url_period'] = $_POST['woo_amazon_url_period'];
				}
				$this->credentials['key'] = $admin_options['amazon_access_key'];
				$this->credentials['secret'] = $admin_options['amazon_access_secret'];
				update_option( $this->settings_name, $admin_options );

				include_once 'templates/settings.php';
			}

			public function wc2_product_download( $file_path, $product_id, $download_id ) {
				// Only run do_shortcode when it is a shortcode and on the front-end, or when it is REST only for GET and context != edit
				$is_shortcode = '[' === substr( $file_path, 0, 1 ) && ']' === substr( $file_path, -1 );
				$is_rest = defined( 'REST_REQUEST' );

				if ( $is_shortcode && (
					( ! $is_rest && ! is_admin() ) ||
					( $is_rest && 'GET' === strtoupper( $_SERVER['REQUEST_METHOD'] ) &&
						( ! isset( $_GET['context'] ) || 'edit' !== $_GET['context'] ) ) ) ) {
					return do_shortcode( $file_path );
				}

				return $file_path;
			}

			/**
			 * Filters the name so the amazon tag or any of its parts don't show - we just want the file name if possible
			 */
			function wc2_product_download_name( $name, $product, $download_id, $file_number ) {
				if ( strpos( $name, '[amazon_s3' ) === false ) {
					return $name;
				}

				$name = str_replace( '[amazon_s3 ', "[amazon_s3 return='name' ", $name );
				return do_shortcode( $name );
			}

			// Kept around for older versions not using wp_remote_get, setting removed
			function set_ssl( $amazon_s3_object ) {
				if ( '1' === $this->disable_ssl ) {
					$amazon_s3_object->disable_ssl_verification();
				}
			}

			function amazon_shortcode( $atts ) {
				require_once 'amazon-s3-api.php';

				extract( shortcode_atts( array(
					'bucket' => '',
					'object' => '',
					'return' => 'url',
					'region' => '',
				), $atts ) );

				if ( 'name' === $return ) {
					return $object;
				}

				$object = str_replace( array( '+', ' ' ), '%20', $object );

				if ( ! empty( $bucket ) && ! empty( $object ) ) {
					$admin_options = get_option( $this->settings_name );
					$period = 60;
					// Check if we should make URL only valid for certain period
					if ( ! empty( $admin_options['amazon_url_period'] ) ) {
						// send time through as seconds
						$period = ( $admin_options['amazon_url_period'] * 60 );
					}

					$s3 = new AmazonS3( $this->credentials );
					$amazon_url = $s3->get_object_url( $bucket, $object, $period, $region );

					if ( ! empty( $amazon_url ) ) {
						return $amazon_url;
					} else {
						$error = __( 'A download failed due to connection problems with Amazon S3, please check your settings.', 'woo_amazon_s3' );
						if ( defined( 'WC_VERSION' ) && WC_VERSION ) {
							wc_add_notice( $error, 'error' );
						} else {
							global $woocommerce;
							$woocommerce->add_error( $error );
						}
					}
				}
			}
		}
	} // End if().
	global $WooCommerce_Amazon_S3_Storage;
	$WooCommerce_Amazon_S3_Storage = new WooCommerce_Amazon_S3_Storage();
}

/**
 * WooCommerce Deactivated Notice.
 */
function wc_amazon_s3_woocommerce_deactivated() {
	/* translators: %s: WooCommerce link */
	echo '<div class="error"><p>' . sprintf( esc_html__( 'WooCommerce Amazon S3 Storage requires %s to be installed and active.', 'wc_amazon_s3' ), '<a href="https://woocommerce.com/" target="_blank">WooCommerce</a>' ) . '</p></div>';
}
