<?php

namespace EasyAffiliate\Controllers;

use EasyAffiliate\Lib\BaseCtrl;
use EasyAffiliate\Lib\Utils;
use EasyAffiliate\Models\Options;

class UpdateCtrl extends BaseCtrl
{
  public function load_hooks() {
    add_filter('pre_set_site_transient_update_plugins', [self::class, 'queue_update']);
    add_filter('plugins_api', [self::class, 'plugin_info'], 11, 3);
    add_action('in_plugin_update_message-easy-affiliate/easy-affiliate.php', [self::class, 'check_incorrect_edition']);
    add_action('esaf_plugin_edition_changed', [self::class, 'clear_update_transients']);
    add_action('admin_notices', [self::class, 'activation_warning']);
    add_action('admin_init', [self::class, 'activate_from_define']);
    add_action('admin_init', [self::class, 'maybe_activate']);
  }

  public static function is_activated() {
    $options = Options::fetch();
    $activated = get_option('esaf_activated');

    return !empty($options->mothership_license) && !empty($activated);
  }

  public static function check_license_activation() {
    $aov = get_option('esaf_activation_override');

    if(!empty($aov)) {
      update_option('esaf_activated', true);
      do_action('esaf_license_activated', ['aov' => 1]);
      return;
    }

    $options = Options::fetch();

    if(empty($options->mothership_license)) {
      return;
    }

    // Only check the key once per day
    $option_key = "esaf_license_check_{$options->mothership_license}";

    if(get_site_transient($option_key)) {
      return;
    }

    $check_count = get_option($option_key, 0) + 1;
    update_option($option_key, $check_count);

    set_site_transient($option_key, true, Utils::hours($check_count > 3 ? 72 : 24));

    $domain = urlencode(Utils::site_domain());
    $args = compact('domain');

    try {
      $act = self::send_mothership_request("/license_keys/check/{$options->mothership_license}", $args);

      if(!empty($act) && is_array($act)) {
        $license_expired = false;

        if(isset($act['expires_at'])) {
          $expires_at = strtotime($act['expires_at']);

          if($expires_at && $expires_at < time()) {
            $license_expired = true;
            update_option('esaf_activated', false);
            do_action('esaf_license_expired', $act);
          }
        }

        if(isset($act['status']) && !$license_expired) {
          if($act['status'] == 'enabled') {
            update_option($option_key, 0);
            update_option('esaf_activated', true);
            do_action('esaf_license_activated', $act);
          }
          elseif($act['status'] == 'disabled') {
            update_option('esaf_activated', false);
            do_action('esaf_license_invalidated', $act);
          }
        }
      }
    }
    catch(\Exception $e) {
      if($e->getMessage() == 'Not Found') {
        update_option('esaf_activated', false);
        do_action('esaf_license_invalidated');
      }
    }
  }

  public static function maybe_activate() {
    $activated = get_option('esaf_activated');

    if(!$activated) {
      self::check_license_activation();
    }
  }

  public static function activate_from_define() {
    $options = Options::fetch();

    if(defined('EASY_AFFILIATE_LICENSE_KEY') && $options->mothership_license != EASY_AFFILIATE_LICENSE_KEY) {
      try {
        if(!empty($options->mothership_license)) {
          // Deactivate the old license key
          self::deactivate_license();
        }

        // If we're using defines then we have to do this with defines too
        $options = Options::fetch();
        $options->edge_updates = false;
        $options->store();

        $act = self::activate_license(EASY_AFFILIATE_LICENSE_KEY);

        $message = $act['message'];
        $callback = function () use($message) {
          require ESAF_VIEWS_PATH . '/shared/errors.php';
        };
      }
      catch(\Exception $e) {
        $error = $e->getMessage();
        $callback = function () use($error) {
          require ESAF_VIEWS_PATH . '/update/activation_warning.php';
        };
      }

      add_action('admin_notices', $callback);
    }
  }

  /**
   * Activate the license with the given key
   *
   * @param string $license_key The license key
   * @return array The license data
   * @throws \Exception If there was an error activating the license
   */
  public static function activate_license($license_key) {
    $options = Options::fetch();

    $args = [
      'domain' => urlencode(Utils::site_domain()),
      'product' => ESAF_EDITION,
    ];

    $act = self::send_mothership_request("/license_keys/activate/{$license_key}", $args, 'post');

    $options->mothership_license = $license_key;
    $options->store();

    $option_key = "esaf_license_check_{$license_key}";
    delete_site_transient($option_key);
    delete_option($option_key);

    delete_site_transient('esaf_update_info');

    do_action('esaf_license_activated_before_queue_update');

    self::manually_queue_update();

    // Clear the add-ons cache
    delete_site_transient('esaf_addons');
    delete_site_transient('esaf_all_addons');

    do_action('esaf_license_activated', $act);

    return $act;
  }

  /**
   * Deactivate the license
   *
   * @return array
   */
  public static function deactivate_license() {
    $options = Options::fetch();
    $license_key = $options->mothership_license;
    $act = ['message' => __('License key deactivated', 'easy-affiliate')];

    if(!empty($options->mothership_license)) {
      try {
        $args = [
          'domain' => urlencode(Utils::site_domain())
        ];

        $act = self::send_mothership_request("/license_keys/deactivate/{$options->mothership_license}", $args, 'post');
      }
      catch (\Exception $e) {
        // Catching here to allow invalid license keys to be deactivated
      }
    }

    $options->mothership_license = '';
    $options->store();

    $option_key = "esaf_license_check_{$license_key}";
    delete_site_transient($option_key);
    delete_option($option_key);

    delete_site_transient('esaf_update_info');

    do_action('esaf_license_deactivated_before_queue_update');

    self::manually_queue_update();

    // Don't need to check the mothership for this one ... we just deactivated
    update_option('esaf_activated', false);

    // Clear the cache of the license and add-ons
    delete_site_transient('wafp_license_info');
    delete_site_transient('esaf_addons');
    delete_site_transient('esaf_all_addons');

    do_action('esaf_license_deactivated', $act);

    return $act;
  }

  public static function queue_update($transient, $force = false) {
    if(empty($transient) || !is_object($transient)) {
      return $transient;
    }

    $options = Options::fetch();

    $update_info = get_site_transient('esaf_update_info');

    if($force || (false === $update_info)) {
      $args = [];

      if($options->edge_updates || (defined('EASY_AFFILIATE_EDGE') && EASY_AFFILIATE_EDGE)) {
        $args['edge'] = 'true';
      }

      if(empty($options->mothership_license)) {
        try {
          // Just here to query for the current version
          $version_info = self::send_mothership_request('/versions/latest/' . ESAF_EDITION, $args);
          $curr_version = $version_info['version'];
          $download_url = '';
        }
        catch(\Exception $e) {
          if(isset($transient->response[ESAF_PLUGIN_SLUG])) {
            unset($transient->response[ESAF_PLUGIN_SLUG]);
          }

          return $transient;
        }
      }
      else {
        try {
          $args['domain'] = urlencode(Utils::site_domain());

          $license_info = self::send_mothership_request("/versions/info/{$options->mothership_license}", $args);
          $curr_version = $license_info['version'];
          $download_url = $license_info['url'];

          set_site_transient('wafp_license_info', $license_info, Utils::hours(24));

          if(Utils::is_incorrect_edition_installed()) {
            $download_url = '';
          }
        }
        catch(\Exception $e) {
          try {
            // Just here to query for the current version
            $version_info = self::send_mothership_request('/versions/latest/' . ESAF_EDITION, $args);
            $curr_version = $version_info['version'];
            $download_url = '';
          }
          catch(\Exception $e) {
            if(isset($transient->response[ESAF_PLUGIN_SLUG])) {
              unset($transient->response[ESAF_PLUGIN_SLUG]);
            }

            self::check_license_activation();
            return $transient;
          }
        }
      }

      set_site_transient(
        'esaf_update_info',
        compact('curr_version', 'download_url'),
        Utils::hours(12)
      );
    }
    else {
      extract($update_info);
    }

    if(isset($curr_version) && version_compare($curr_version, ESAF_VERSION, '>')) {
      global $wp_version;

      $transient->response[ESAF_PLUGIN_SLUG] = (object) [
        'id' => ESAF_PLUGIN_SLUG,
        'slug' => ESAF_PLUGIN_NAME,
        'plugin' => ESAF_PLUGIN_SLUG,
        'new_version' => $curr_version,
        'url' => 'https://easyaffiliate.com/',
        'package' => $download_url,
        'tested' => $wp_version
      ];
    }
    else {
      unset($transient->response[ESAF_PLUGIN_SLUG]);

      // Enables the "Enable auto-updates" link
      $transient->no_update[ESAF_PLUGIN_SLUG] = (object) [
        'id' => ESAF_PLUGIN_SLUG,
        'slug' => ESAF_PLUGIN_NAME,
        'plugin' => ESAF_PLUGIN_SLUG,
        'new_version' => ESAF_VERSION,
        'url' => 'https://easyaffiliate.com/',
        'package' => ''
      ];
    }

    self::check_license_activation();
    return $transient;
  }

  public static function manually_queue_update() {
    $transient = get_site_transient("update_plugins");
    set_site_transient("update_plugins", self::queue_update($transient, true));
  }

  public static function plugin_info($api, $action, $args) {
    global $wp_version;

    if(!isset($action) || $action != 'plugin_information') {
      return $api;
    }

    if(!isset($args->slug) || $args->slug != ESAF_PLUGIN_NAME) {
      return $api;
    }

    $options = Options::fetch();
    $args = [];

    if($options->edge_updates || (defined('EASY_AFFILIATE_EDGE') && EASY_AFFILIATE_EDGE)) {
      $args['edge'] = 'true';
    }

    if(empty($options->mothership_license)) {
      try {
        // Just here to query for the current version
        $version_info = self::send_mothership_request('/versions/latest/' . ESAF_EDITION, $args);
        $curr_version = $version_info['version'];
        $version_date = $version_info['version_date'];
        $download_url = '';
      }
      catch(\Exception $e) {
        return $api;
      }
    }
    else {
      try {
        $args['domain'] = urlencode(Utils::site_domain());

        $license_info = self::send_mothership_request("/versions/info/{$options->mothership_license}", $args);
        $curr_version = $license_info['version'];
        $version_date = $license_info['version_date'];
        $download_url = $license_info['url'];

        if(Utils::is_incorrect_edition_installed()) {
          $download_url = '';
        }
      }
      catch(\Exception $e) {
        try {
          // Just here to query for the current version
          $version_info = self::send_mothership_request('/versions/latest/' . ESAF_EDITION, $args);
          $curr_version = $version_info['version'];
          $version_date = $version_info['version_date'];
          $download_url = '';
        }
        catch(\Exception $e) {
          return $api;
        }
      }
    }

    return (object) [
      'slug' => ESAF_PLUGIN_NAME,
      'name' => esc_html(ESAF_DISPLAY_NAME),
      'author' => ESAF_AUTHOR,
      'author_profile' => ESAF_AUTHOR_URI,
      'contributors' => [
        'caseproof' => [
          'profile' => ESAF_AUTHOR_URI,
          'avatar' => 'https://secure.gravatar.com/avatar/762b61e36276ff6dc0d7b03b8c19cfab?s=96&d=monsterid&r=g',
          'display_name' => ESAF_AUTHOR_NAME
        ]
      ],
      'homepage' => 'https://easyaffiliate.com/',
      'version' => $curr_version,
      'new_version' => $curr_version,
      'requires' => '5.2',
      'requires_php' => ESAF_MINIMUM_PHP_VERSION,
      'tested' => $wp_version,
      'compatibility' => [$wp_version => [$curr_version => [100, 0, 0]]],
      'rating' => '100.00',
      'num_ratings' => '1',
      'added' => '2012-12-02',
      'last_updated' => $version_date,
      'tags' => [
        'affiliate' => 'affiliate',
        'affiliate program' => 'affiliate program',
      ],
      'sections' => [
        'description' => '<p>' . ESAF_DESCRIPTION . '</p>',
        'faq' => '<p>' . sprintf(esc_html__('You can access in-depth information about Easy Affiliate at %1$sthe Easy Affiliate User Manual%2$s.', 'easy-affiliate'), '<a href="https://easyaffiliate.com/docs/">', '</a>') . '</p>',
        'changelog' => '<p>' . sprintf(esc_html__('See the %1$splugin changelog%2$s.', 'easy-affiliate'), '<a href="https://easyaffiliate.com/change-log/">', '</a>') . '</p>'
      ],
      'download_link' => $download_url
    ];
  }

  public static function send_mothership_request($endpoint, $args = [], $method = 'get', $blocking = true) {
    if(defined('ESAF_MOTHERSHIP_DOMAIN')) {
      $domain = ESAF_MOTHERSHIP_DOMAIN;
    }
    elseif(apply_filters('esaf_secure_mothership_request', true)) {
      $domain = 'https://mothership.caseproof.com';
    }
    else {
      $domain = 'http://mothership.caseproof.com';
    }

    $uri = "{$domain}{$endpoint}";

    $arg_array = [
      'method'    => strtoupper($method),
      'body'      => $args,
      'timeout'   => 15,
      'blocking'  => $blocking
    ];

    $resp = wp_remote_request($uri, $arg_array);

    // If we're not blocking then the response is irrelevant, so we'll just return true.
    if(!$blocking) {
      return true;
    }

    if(is_wp_error($resp)) {
      throw new \Exception(__('You had an HTTP error connecting to Caseproof\'s Mothership API', 'easy-affiliate'));
    }
    else {
      if(null !== ($json_res = json_decode($resp['body'], true))) {
        if(isset($json_res['error'])) {
          throw new \Exception($json_res['error']);
        }
        else {
          return $json_res;
        }
      }
      else {
        throw new \Exception(__('Your License Key was invalid', 'easy-affiliate'));
      }
    }
  }

  public static function activation_warning() {
    $options = Options::fetch();

    if(!self::is_activated()) {
      require ESAF_VIEWS_PATH . '/update/activation_warning.php';
    }
  }

  public static function addons($return_object = false, $force = false, $all = false) {
    $options = Options::fetch();
    $license = $options->mothership_license;
    $transient = $all ? 'esaf_all_addons' : 'esaf_addons';

    if($force) {
      delete_site_transient($transient);
    }

    if($addons = get_site_transient($transient)) {
      $addons = json_decode($addons);
    }
    else {
      $addons = [];

      if(!empty($license)) {
        try {
          $domain = urlencode(Utils::site_domain());
          $args = compact('domain');

          if($all) {
            $args['all'] = 'true';
          }

          if($options->edge_updates || (defined("EASY_AFFILIATE_EDGE") && EASY_AFFILIATE_EDGE)) {
            $args['edge'] = 'true';
          }

          $addons = self::send_mothership_request('/versions/addons/' . ESAF_EDITION . "/{$license}", $args);
        }
        catch(\Exception $e) {
          // fail silently
        }
      }

      $json = json_encode($addons);
      set_site_transient($transient, $json, Utils::hours(12));

      if($return_object) {
        $addons = json_decode($json);
      }
    }

    return $addons;
  }

  public static function check_incorrect_edition() {
    if(Utils::is_incorrect_edition_installed()) {
      printf(
        /* translators: %1$s: open link tag, %2$s: close link tag */
        ' <strong>' . esc_html__('To restore automatic updates, %1$sinstall the correct edition%2$s of Easy Affiliate.', 'easy-affiliate') . '</strong>',
        sprintf('<a href="%s">', esc_url(admin_url('admin.php?page=easy-affiliate-settings'))),
        '</a>'
      );
    }
  }

  public static function clear_update_transients() {
    delete_site_transient('update_plugins');
    delete_site_transient('esaf_update_info');
    delete_site_transient('esaf_addons');
    delete_site_transient('esaf_all_addons');
  }
} //End class
