<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/*
 * Plugin Name: Outrank
 * Plugin URI: https://outrank.so
 * Description: Get traffic and outrank competitors with automatic SEO-optimized content generation published to your WordPress site.
 * Version: 1.0.2
 * Author: Outrank
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Requires PHP: 8.0
 * Requires at least: 6.4
 * Tested up to: 6.8
*/

define('OUTRANK_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('OUTRANK_PLUGIN_URL', plugin_dir_url(__FILE__));
require_once plugin_dir_path(__FILE__) . 'includes/image-functions.php';

// Add admin menu pages
add_action('admin_menu', 'outrank_add_outrank_menu');
function outrank_add_outrank_menu() {
    add_menu_page(
        'Outrank Menu',
        'Outrank',
        'manage_options',
        'outrank',
        'outrank_page',
        'data:image/svg+xml;base64,' . base64_encode(file_get_contents(OUTRANK_PLUGIN_PATH . 'images/icon.svg')),
        60
    );
    add_submenu_page('outrank', 'Home', 'Home', 'manage_options', 'outrank', 'outrank_page');
    add_submenu_page('outrank', 'Manage', 'Manage', 'manage_options', 'outrank_manage', 'outrank_manage_page');
}

// Redirect to manage page if no API key is set
add_action('admin_init', 'outrank_check_api_key_redirect');
function outrank_check_api_key_redirect() {
    // Only redirect if we're on the Outrank home page
    if (isset($_GET['page']) && $_GET['page'] === 'outrank') {
        $apiKey = get_option('outrank_api_key');
        if (empty($apiKey)) {
            wp_safe_redirect(admin_url('admin.php?page=outrank_manage'));
            exit;
        }
    }
}

// Handle activation redirect
add_action('admin_init', 'outrank_activation_redirect');
function outrank_activation_redirect() {
    // Only redirect if transient exists
    if (get_transient('outrank_activation_redirect')) {
        delete_transient('outrank_activation_redirect');
        
        // Don't redirect on multi-site activations or bulk plugin activations
        if (is_network_admin() || isset($_GET['activate-multi'])) {
            return;
        }
        
        // Redirect to manage page
        wp_safe_redirect(admin_url('admin.php?page=outrank_manage'));
        exit;
    }
}

// Include admin pages
function outrank_page() {
    include_once OUTRANK_PLUGIN_PATH . 'pages/home.php';
}

function outrank_manage_page() {
    include_once OUTRANK_PLUGIN_PATH . 'pages/manage.php';
}

// Activation hook: Create custom table
register_activation_hook(__FILE__, 'outrank_activate');
function outrank_activate() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'outrank_manage';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "
    CREATE TABLE {$table_name} (
        id INT(11) NOT NULL AUTO_INCREMENT,
        image TEXT NOT NULL,
        slug VARCHAR(191) NOT NULL,
        title TEXT NOT NULL,
        meta_description TEXT NOT NULL,
        status VARCHAR(50) NOT NULL,
        created_at DATETIME NOT NULL,
        PRIMARY KEY (id),
        UNIQUE KEY slug_unique (slug)
    ) ENGINE=InnoDB $charset_collate;
    ";

    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    dbDelta($sql);
    outrank_schedule_cron();
    
    // Set transient for activation redirect
    set_transient('outrank_activation_redirect', true, 30);
}

// Deactivation hook: Keep all data
register_deactivation_hook(__FILE__, 'outrank_deactivate');

function outrank_deactivate() {
    // Clear scheduled cron events
    $timestamp = wp_next_scheduled('outrank_daily_sync');
    if ($timestamp) {
        wp_unschedule_event($timestamp, 'outrank_daily_sync');
    }
    
    // Note: We intentionally keep all data (database table, options, posts)
    // This allows users to reactivate without losing anything
}

// Enqueue admin styles and scripts
add_action('admin_enqueue_scripts', 'outrank_add_plugin_assets');
function outrank_add_plugin_assets($hook_suffix = '') {
    if (strpos($hook_suffix, 'outrank') === false) return; // Only enqueue on outrank pages

    wp_enqueue_style('outrank-style', OUTRANK_PLUGIN_URL . 'css/manage.css', [], '1.0.0');
    wp_enqueue_style('outrank-home-style', OUTRANK_PLUGIN_URL . 'css/home.css', [], '1.0.0');

    wp_enqueue_script('outrank-script', OUTRANK_PLUGIN_URL . 'script/manage.js', ['jquery'], '1.0.0', true);

    // Localize ajax data only on Manage page
    if ($hook_suffix === 'toplevel_page_outrank_manage' || $hook_suffix === 'outrank_page_outrank_manage') {
        wp_localize_script('outrank-script', 'outrankAjax', [
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce'   => wp_create_nonce('outrank_fetch_nonce'),
        ]);
    }
}

// Schedule daily cron event
function outrank_schedule_cron() {
    if (!wp_next_scheduled('outrank_daily_sync')) {
        wp_schedule_event(time(), 'daily', 'outrank_daily_sync');
    }
}
add_action('outrank_daily_sync', 'outrank_fetch_articles');

// Clear scheduled cron event
// function outrank_clear_cron() {
//     $timestamp = wp_next_scheduled('outrank_daily_sync');
//     if ($timestamp) {
//         wp_unschedule_event($timestamp, 'outrank_daily_sync');
//     }
// }

// AJAX handler for manual fetching articles
add_action('wp_ajax_outrank_fetch_articles_now', function () {
    if (!check_ajax_referer('outrank_fetch_nonce', 'security', false)) {
        wp_send_json_error(['message' => 'Invalid nonce']);
    }
    outrank_fetch_articles();
    wp_send_json_success(['message' => '✅ Articles fetched successfully']);
});

// Fetch articles from external API and insert into DB + create WP pages
function outrank_fetch_articles() {
    global $wpdb;
    $apiKey = get_option('outrank_api_key');
    if (!$apiKey) {
    //     if ( defined('WP_DEBUG') && WP_DEBUG === true){
    //         error_log('Outrank: API key missing');
    //     }
        return;
    }

    $response = wp_remote_get("https://www.outrank.so/api/integrations/framer/articles", [
        'headers' => ['X-API-Key' => $apiKey],
        'timeout' => 60,
    ]);

    if (is_wp_error($response) && defined('WP_DEBUG') && WP_DEBUG === true) {
    //     error_log('Outrank fetch error: ' . $response->get_error_message());
        return;
    }

    $body = wp_remote_retrieve_body($response);
    $data = json_decode($body, true);

    if (empty($data['data']['articles']) && defined('WP_DEBUG') && WP_DEBUG === true) {
    //     error_log('Outrank: No articles found in response.');
        return;
    }

    $table = $wpdb->prefix . 'outrank_manage';
    $slugCounts = [];

    foreach ($data['data']['articles'] as $article) {
        $baseSlug = !empty($article['slug']) ? sanitize_title($article['slug']) : sanitize_title($article['title'] ?? 'untitled');
        $slug = $baseSlug;
        $slugCounts[$baseSlug] = ($slugCounts[$baseSlug] ?? 0) + 1;
        if ($slugCounts[$baseSlug] > 1) {
            $slug .= '-' . $slugCounts[$baseSlug];
        }

        $cache_key = 'outrank_slug_exists_' . md5($slug);
        $exists = wp_cache_get($cache_key);
        
        if ($exists === false) {
            $table_name = esc_sql($wpdb->prefix . 'outrank_manage');
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
            $exists = $wpdb->get_var(
                $wpdb->prepare("SELECT COUNT(*) FROM %i WHERE slug = %s", $table_name, $slug)
            );
            wp_cache_set($cache_key, $exists, '', 600); // cache for 10 minutes
        }
        if ($exists > 0) {
            continue; // skip duplicates
        }

        $created_at = !empty($article['created_at']) ? gmdate('Y-m-d H:i:s', strtotime($article['created_at'])) : current_time('mysql');

        // Insert into DB
        $imageId = outrank_upload_image_from_url($article['image_url'] ?? '');

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
        $inserted = $wpdb->insert(
            $table_name,
            [
                'image' => $imageId,
                'slug'             => sanitize_text_field($slug),
                'title'            => sanitize_text_field($article['title']),
                'meta_description' => sanitize_text_field($article['meta_description'] ?? ''),
                'status'           => get_option('outrank_post_as_draft', 'yes') === 'yes' ? 'draft' : 'publish',
                'created_at'       => $created_at,
            ]
        );


        if ($inserted === false && defined('WP_DEBUG') && WP_DEBUG === true) {
        //     error_log('Outrank DB Insert failed: ' . $wpdb->last_error);
            continue;
        }

        // Handle categories properly
        $category_ids = [];
        if (!empty($article['category'])) {
            if (is_array($article['category'])) {
                foreach ($article['category'] as $cat_name) {
                    $cat = get_category_by_slug(sanitize_title($cat_name));
                    if (!$cat) {
                        $cat_id = wp_create_category($cat_name);
                        $category_ids[] = $cat_id;
                    } else {
                        $category_ids[] = $cat->term_id;
                    }
                }
            } else {
                $cat = get_category_by_slug(sanitize_title($article['category']));
                if (!$cat) {
                    $cat_id = wp_create_category($article['category']);
                    $category_ids[] = $cat_id;
                } else {
                    $category_ids[] = $cat->term_id;
                }
            }
        } else {
            // Default category if none provided
            $default_cat = get_category_by_slug('outrank');
            if (!$default_cat) {
                $cat_id = wp_create_category('Outrank');
                $category_ids[] = $cat_id;
            } else {
                $category_ids[] = $default_cat->term_id;
            }
        }

        // Handle author properly
        $author_id = 1; // Default to admin
        if (!empty($article['author'])) {
            if (is_numeric($article['author'])) {
                $author_id = (int) $article['author'];
            } else {
                $user = get_user_by('login', $article['author']);
                if ($user) {
                    $author_id = $user->ID;
                }
            }
        }

        // Prepare post data
        $post_data = [
            'post_title'    => sanitize_text_field($article['title']),
            'post_content'  => wp_kses_post($article['content']),
            'post_status'   => get_option('outrank_post_as_draft', 'yes') === 'yes' ? 'draft' : 'publish',
            'post_type'     => 'post',
            'post_name'     => $slug,
            'post_category' => $category_ids,
            'post_author'   => $author_id,
            'post_date'     => $created_at,
        ];

        // Handle tags
        if (!empty($article['tags'])) {
            if (is_array($article['tags'])) {
                $post_data['tags_input'] = $article['tags'];
            } else {
                $post_data['tags_input'] = explode(',', $article['tags']);
            }
        }

        // Insert as WP post
        $post_id = wp_insert_post($post_data);//db call ok
        
        if (is_wp_error($post_id) && defined('WP_DEBUG') && WP_DEBUG === true) {
            // error_log('Outrank WP Post Insert failed: ' . $post_id->get_error_message());
            continue;
        } elseif ($post_id === 0 && defined('WP_DEBUG') && WP_DEBUG === true) {
            // error_log('Outrank WP Post Insert failed: wp_insert_post returned 0');
            continue;
        }

        // Set featured image if available
        if (!empty($imageId)) {
            set_post_thumbnail($post_id, $imageId);
        }

        // Set SEO meta data for popular SEO plugins
        if (!empty($article['meta_description'])) {
            $meta_description = sanitize_text_field($article['meta_description']);
            
            // Yoast SEO
            update_post_meta($post_id, '_yoast_wpseo_metadesc', $meta_description);
            
            // Rank Math
            update_post_meta($post_id, 'rank_math_description', $meta_description);
            
            // All in One SEO
            update_post_meta($post_id, '_aioseo_description', $meta_description);
            
            // SEOPress
            update_post_meta($post_id, '_seopress_titles_desc', $meta_description);
        }
        
        // Set focus keyphrase/keyword if provided
        if (!empty($article['focus_keyword']) || !empty($article['focus_keyphrase'])) {
            $focus_keyword = sanitize_text_field($article['focus_keyword'] ?? $article['focus_keyphrase'] ?? '');
            
            // Yoast SEO
            update_post_meta($post_id, '_yoast_wpseo_focuskw', $focus_keyword);
            
            // Rank Math
            update_post_meta($post_id, 'rank_math_focus_keyword', $focus_keyword);
            
            // All in One SEO (stores as JSON)
            $aioseo_keyphrases = json_encode([
                ['keyphrase' => $focus_keyword, 'score' => 0]
            ]);
            update_post_meta($post_id, '_aioseo_keyphrases', $aioseo_keyphrases);
            
            // SEOPress
            update_post_meta($post_id, '_seopress_analysis_target_kw', $focus_keyword);
        }
        
        // Set SEO title using the normal title
        if (!empty($article['title'])) {
            $seo_title = sanitize_text_field($article['title']);
            
            // Yoast SEO
            update_post_meta($post_id, '_yoast_wpseo_title', $seo_title);
            
            // Rank Math
            update_post_meta($post_id, 'rank_math_title', $seo_title);
            
            // All in One SEO
            update_post_meta($post_id, '_aioseo_title', $seo_title);
            
            // SEOPress
            update_post_meta($post_id, '_seopress_titles_title', $seo_title);
        }

        // if (defined('WP_DEBUG') && WP_DEBUG === true) {
            // error_log("Outrank: Successfully created post ID: $post_id with title: " . $article['title']);
        // }
    }
}

// Helper function to get all articles from DB
function outrank_get_articles() {
    global $wpdb;

    $cache_key = 'outrank_all_articles';
    $articles = wp_cache_get($cache_key);

    if ($articles === false) {
        $table_name = esc_sql($wpdb->prefix . 'outrank_manage');
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
        $articles = $wpdb->get_results($wpdb->prepare("SELECT * FROM %i ORDER BY created_at DESC", $table_name));
        wp_cache_set($cache_key, $articles, '', 300); // Cache for 5 minutes
    }

    return $articles;
}

require_once OUTRANK_PLUGIN_PATH . 'libs/api.php';

$api_file = OUTRANK_PLUGIN_PATH . 'libs/api.php';

if (file_exists($api_file)) {
    require_once $api_file;
    // if (defined('WP_DEBUG') && WP_DEBUG === true) {
    //     error_log("✅ api.php included from $api_file");
    // }
// } else {
    // if (defined('WP_DEBUG') && WP_DEBUG === true) {
    //     error_log("❌ api.php NOT found at $api_file");
    // }
}