<?php

namespace EasyAffiliate\Lib;

abstract class Migrator implements MigratorInterface {
  /**
   * Array of logs to display for non-critical issues.
   *
   * @var string[]
   */
  protected $logs = [];

  /**
   * Runs before the migrator starts.
   */
  protected static function before_start() {
    if(function_exists('set_time_limit')) {
      set_time_limit(0);
    }

    wp_defer_term_counting(true);
    wp_defer_comment_counting(true);
    wp_suspend_cache_invalidation();
  }

  /**
   * Runs when the migrator finishes.
   */
  protected static function finish() {
    wp_suspend_cache_invalidation(false);
    wp_cache_flush();
    wp_defer_term_counting(false);
    wp_defer_comment_counting(false);
  }

  /**
   * Free memory, to prevent memory exhausted errors.
   */
  public static function free_memory() {
    global $wpdb, $wp_object_cache;

    // Empty the query log, this can grow constantly if SAVEQUERIES is enabled
    $wpdb->queries = [];

    if($wp_object_cache instanceof \WP_Object_Cache) {
      if(function_exists('wp_cache_flush_runtime')) {
        wp_cache_flush_runtime();
      }
      elseif(!wp_using_ext_object_cache() || apply_filters('esaf_migrate_flush_cache', false)) {
        wp_cache_flush();
      }
    }
  }

  /**
   * Get the limit and offset from the given request data (if any).
   *
   * @param  array $data          The data for the current request.
   * @param  int   $default_limit The default limit to use if none exists in the request data.
   * @return array
   */
  protected function get_request_limit_offset(array $data, int $default_limit): array {
    $limit = isset($data['limit']) && is_numeric($data['limit']) && $data['limit'] > 0 ? Utils::clamp((int) $data['limit'], 1, 500) : $default_limit;
    $offset = isset($data['offset']) && is_numeric($data['offset']) && $data['offset'] > 0 ? (int) $data['offset'] : 0;

    return [$limit, $offset];
  }
}
