<?php

namespace EasyAffiliate\Controllers;

use EasyAffiliate\Lib\BaseCtrl;
use EasyAffiliate\Lib\Cookie;
use EasyAffiliate\Lib\Track;
use EasyAffiliate\Lib\Utils;
use EasyAffiliate\Models\Options;
use EasyAffiliate\Models\Transaction;
use EasyAffiliate\Models\User;

/** This is a special controller that handles all of the WishList Member specific
  * public static functions for the Affiliate Program.
  */
class WishListMemberCtrl extends BaseCtrl {

  public function load_hooks() {
  $options = Options::fetch();
    if(!in_array('wishlist_member', $options->integration) || !self::is_plugin_active()) {
      return;
    }
    add_action( 'wishlistmember_finish_incomplete_registration', [$this, 'track_for_auto_completed_accounts'], 9999 );
    add_action( 'wishlistmember_existing_member_purchase', [$this, 'track_for_auto_completed_accounts'], 9999 );
    add_action( 'wishlistmember_shoppingcart_rebill', [$this, 'process_recurring'], 10 );
    add_action( 'wishlistmember_shoppingcart_register', [$this, 'track_transacton_for_incomplete_reg'], 10 );
  }

  /**
   * Check WishList Member active.
   *
   * @return bool
   */
  public static function is_plugin_active() {
    global $WishListMemberInstance;

    return isset( $WishListMemberInstance ) ? true : false;

  }

  /**
   * Create a record for tracking the affiliate once an incomplete registration has been created.
   * This happens after the user completes the payment in the Shopping Cart (eg. Stripe, Paypal, Clickbank)
   * and is redirected back to the WLM's registration page to complete the registration.
   * Since WishList Member doesn't record payment/purchase transactions we'll just
   * save a separate data in the WLM's option table so we can track the affiliate_id in case there's any and use it for recurring events.
   */
  public function track_transacton_for_incomplete_reg() {

    $level_id         = wlm_post_data()['sku'];
    $level_id         = ( $level_id ) ? $level_id: wlm_post_data()['wpm_id'];

    $paid_amount      = wlm_post_data()['paid_amount'];
    $transaction_type = 'Initial';
    $sc               = wlm_post_data()['sc_type'];

    // Let's get the transaction ID based on temporary username and the level's ID.
    $user_data = wlmapi_get_member_by( 'user_login', wlm_post_data()['username'] );
    $user_id   = $user_data['members']['member'][0]['id'];
    $member_data = wlmapi_get_level_member_data( $level_id, $user_id );

    $txn_id      = $member_data['member']['level']->TxnID;

    $this->process_tracking( $level_id, $txn_id, $paid_amount, $transaction_type, $sc );
  }

  /**
   * This function is use to process accounts that are auto completed.
   * When the "Enable Auto-Created Accounts for Integrations" is enabled for the level and there's no set delay
   * the user is not redirected to the registration page and the hook wishlistmember_registration_page is not triggered.
   * Adding a separate tracker for that using this function.
   */
  public function track_for_auto_completed_accounts( $data ) {

    $level_id         = wlm_post_data()['sku'];
    $level_id         = ( $level_id ) ? $level_id: wlm_post_data()['wpm_id'];
    $paid_amount      = wlm_post_data()['paid_amount'];
    $transaction_type = 'Initial';
    $sc               = wlm_post_data()['sc_type'];
    $txn_id           = wlm_post_data()['sctxnid'];

    // Let's get the transaction ID based on email address and the level's ID.
    $user_data = wlmapi_get_member_by( 'user_email', wlm_post_data()['email']);
    $user_id   = $user_data['members']['member'][0]['id'];
    $member_data = wlmapi_get_level_member_data( $level_id, $user_id );

    if ( !$txn_id ) {
      $txn_id      = $member_data['member']['level']->TxnID;
    }

    if ( ! $user_id || ! $txn_id ) {
      return;
    }

    // Let's save the affiliate ID to this user in case there are future recurring payments.
    $customer = new User($user_id);
    $customer->referrer = (int) Cookie::get_affiliate_id();
    $customer->store();

    $this->process_tracking( $level_id, $txn_id, $paid_amount, $transaction_type, $sc );
  }

  /**
   * Processes the tracking/giving out the commission.
   * @param int $level_id - SKU/ID of the membership level
   * @param string $txn_id - Transaction ID
   * @param int $paid_amount - The amount the user paid
   * @param string $transaction_type - Initial or Recurring
   * @param string $sc - The Name of the Shopping Cart that initiated this transaction
   * @param int $user_id - The ID of the customer
   */
  public function process_tracking( $level_id, $txn_id, $paid_amount, $transaction_type, $sc, $user_id = '' ) {

    // Make sure not to track an admin manually adding/registering a user
    if(is_super_admin()) {
      return;
    }

    if( ! $txn_id ) {
      return;
    }

    // Check first if there's already a transaction made for this using the $txn_id, if so just return.
    $existing_transaction = Transaction::get_one(['source' => 'wishlist_member', 'trans_num' => $txn_id]);

    if( $existing_transaction instanceof Transaction ) {
      return;
    }

    // Save the data in the WLM's options table so we can track use it in case there are recurring payments..
    // Make sure there's an affiliate before we track this.
    $affiliate_id = Cookie::get_affiliate_id();
    $click_id     = Cookie::get_click_id();

    if( $affiliate_id > 0 ) {
      $data = maybe_serialize(
        array(
          'level_id'        => $level_id,
          'affiliate_id'    => $affiliate_id,
          'click_id'        => $click_id,
          'initial_price'   => $paid_amount
        )
      );

      // Let's get the name of the Membership Level.
      $wpm_levels = wishlistmember_instance()->get_option( 'wpm_levels' );
      $level_name = $wpm_levels[$level_id]['name'];
      $trans_num  = $txn_id;
      if ( 'Stripe' == $sc ) {
        // For Stripe Let's use the Stripe Charge ID for the transaction ID (appears as Invoice in EA) that we pass to EA.
        $trans_num = wlm_post_data()['stripe_charge_id'];
      }

      Track::sale(
        'wishlist_member',
        $paid_amount,
        $trans_num,
        $level_id,
        $sc . ' - ' .$level_name,
        $level_id,
        '',
        $user_id,
        '',
        1
      );

      wishlistmember_instance()->save_option( 'easy_affiliate_'.$txn_id, $data );
    }
  }

  /**
   * Processes possible recurring transactions.
   */
  public function process_recurring() {

    // Don't process if it's not a supported SC rebill.
    if ( ! wlm_post_data()['is_wlm_sc_rebill'] ) {
      return;
    }

    // Get transaction ID from POST data.
    $txn_id = wlm_post_data()['sctxnid'];

    // Check if there's a user that has that transaction ID.
    $user = wishlistmember_instance()->get_user_id_from_txn_id( $txn_id );
    if( ! $user ) {
      return;
    }

    // Check if there's any entry on the WLM options table that matches the transaction ID.
    $track_info = wishlistmember_instance()->get_option( 'easy_affiliate_'.$txn_id );
    if( ! $track_info ) {
      return;
    }

    // Get the Level's ID and affiliate ID from $track_info.
    // We want to add the commission to the previously saved affiliate ID.
    $level_id     = $track_info['level_id'];
    $affiliate_id = $track_info['affiliate_id'];
    $click_id     = $track_info['click_id'];
    $sc           = wlm_post_data()['sc_type'];
    $paid_amount  = wlm_post_data()['paid_amount'];

    if( $paid_amount <= 0 ) {
      return;
    }

    // Process the commission using the affiliate ID Saved as the Referrer on the User's Meta.
    // Let's get the name of the Membership Level.
    $wpm_levels = wishlistmember_instance()->get_option( 'wpm_levels' );
    $level_name = $wpm_levels[$level_id]['name'];

    add_filter(
      'esaf_modify_cookie_affiliate_id',
      function() use( $affiliate_id ) {
        return $affiliate_id;
      }
    );

    add_filter(
      'esaf_modify_cookie_click_id',
      function() use( $click_id ) {
        return $click_id;
      }
    );

    // Get txn_count from saved data. If no txn_count yet then it's considered the first one.
    $txn_count = isset( $track_info['txn_count'] ) ? (int) $track_info['txn_count'] : 0;
    $txn_count = $txn_count + 1;

    if ( 'Stripe' == $sc ) {
      // For Stripe Let's save the Stripe Charge ID for the trans_num
      $trans_num = wlm_post_data()['stripe_charge_id'];
    } else {
      // For other integration such as Paypal Standard and Paypal Checkout let's use the original txn_id and just add the count.
      $trans_num = $sc . '-rebill-' .$txn_id . '-'.$txn_count;
    }

    Track::sale(
      'wishlist_member',
      $paid_amount,
      $trans_num,
      $level_id,
      sprintf(
        // Translators: %1$s = payment gateway name; %2$s = membership level name.
        __( '%1$s - Recurring payment for %2$s', 'easy-affiliate'),
        wlm_post_data()['sc_type'],
        $level_name
      ),
      $level_id,
      '',
      $user,
      '',
      $txn_count
    );

    // Update txn_count of the original transaction.
    $track_info['txn_count'] = $txn_count;
    wishlistmember_instance()->save_option( 'easy_affiliate_'.$txn_id, $track_info );
  }
}
