<?php

namespace EasyAffiliate\Controllers;

use EasyAffiliate\Lib\BaseCtrl;
use EasyAffiliate\Lib\Cookie;
use EasyAffiliate\Lib\Track;
use EasyAffiliate\Models\Options;
use EasyAffiliate\Models\Transaction;
use EasyAffiliate\Models\User;

/** Handles all the integration hooks into MemberMouse */

class MemberMouseCtrl extends BaseCtrl {

  public function load_hooks() {
    $options = Options::fetch();
    if(!in_array('membermouse', $options->integration) || !self::is_plugin_active()) {
      return;
    }
    add_action('mm_payment_received', [$this, 'track_payment_received']);
    add_action('mm_payment_rebill', [$this, 'track_payment_rebill']);
    add_action('mm_product_purchase', [$this, 'store_affiliate_user_data']);
    add_action('mm_refund_issued', [$this, 'track_refund']);
    add_action('mm_subscription_options_section_after', [$this, 'add_coupon_associate_metabox']);
    add_action('membermouse_enhancedatagrid_scripts', [$this, 'enqueue_scripts']);
    add_action('mm_save_coupon_after', [$this, 'maybe_save_coupon_affiliate']);
  }

  /**
   * @param \MM_Coupon $coupon
   */
  public function maybe_save_coupon_affiliate($coupon) {
    if (!isset($_POST['esaf_associate_affiliate_user'])) {
      return;
    }

    $affiliate_user = get_user_by('login', $_POST['esaf_associate_affiliate_user']);

    if ($affiliate_user instanceof \WP_User) {
      delete_metadata('user', $affiliate_user->ID, 'esaf_associate_affiliate_coupon_' . $coupon->getId(), null, true);
      update_user_meta($affiliate_user->ID, 'esaf_associate_affiliate_coupon_' . $coupon->getId(), 1);
    }
  }

  protected function get_meta_value_by_key($meta_key) {
    global $wpdb;

    return $wpdb->get_results($wpdb->prepare( "SELECT user_id FROM $wpdb->usermeta WHERE meta_key = %s", $meta_key) , ARRAY_A  );
  }

  public function enqueue_scripts($view) {
    if ( ! class_exists(\MM_CouponView::class) || ! $view instanceof \MM_CouponView) {
      return;
    }

    wp_enqueue_script('suggest');
  }

  protected function get_affiliate_user_from_coupon_id($coupon_id, $return_object = false) {
    $affiliate_users = $this->get_meta_value_by_key('esaf_associate_affiliate_coupon_' . $coupon_id);

    foreach ($affiliate_users as $user) {
      $wp_user = new \WP_User($user['user_id']);

      if ($return_object) {
          return $wp_user;
      }

      return $wp_user->user_login;
    }

    return null;
  }

  public function add_coupon_associate_metabox($coupon) {
      $coupon_id = $coupon->getId();
      $user_login = '';

      if (!empty($coupon_id)) {
          $user_login = $this->get_affiliate_user_from_coupon_id($coupon_id);
      }
    ?>
      <tr>
          <td colspan="2">
              <div style="width: 98%; margin-top: 8px;" class="mm-divider"></div>
          </td>
      </tr>
      <tr>
          <td width="140" style="vertical-align:top;"><?php echo _mmt("Affiliate"); ?></td>
          <td>
              <div id="mm_affiliate_options_section">
                  <label for="esaf-associate-affiliate-user"><?php echo esc_html(__('Associate an Affiliate', 'easy-affiliate')); ?></label>
                  <input type="text" name="esaf_associate_user" id="esaf-associate-affiliate-user" value="<?php echo esc_attr($user_login); ?>" placeholder="<?php echo esc_attr(__('leave blank if not applicable', 'easy-affiliate')); ?>">
              </div>
          </td>
      </tr>
    <script>
        jQuery(document).ready(function ($) {
            $('#esaf-associate-affiliate-user').suggest(
                ajaxurl + '?action=wafp_affiliate_search',
                {
                    delay: 500,
                    minchars: 2
                }
            );
        });
    </script>
    <?php
  }

  public function log($data) {
    if (defined('WP_ESAF_DEBUG') && WP_ESAF_DEBUG) {
      file_put_contents(WP_CONTENT_DIR . '/membermouse-ea.log', date('Y-m-d H:i:s') . ' ' . print_r($data, true) . PHP_EOL, FILE_APPEND);
    }
  }

  public function store_affiliate_user_data($orderData) {
    $this->log('Order submitted');
    $this->log($orderData);
    $affiliate_id = Cookie::get_affiliate_id();
    $this->log($affiliate_id);

    if (empty($affiliate_id)) {
      return;
    }

    $user_id = isset($orderData['member_id']) ? $orderData['member_id'] : '';
    $user = new User($user_id);
    $user->referrer = $affiliate_id;

    if ($user_id > 0) {
      $user->store();
    }
  }

  protected function maybe_override_customer_referrer_with_coupon_affiliate($coupon_id) {
    /** @var \WP_User $affiliate */
    $affiliate = $this->get_affiliate_user_from_coupon_id($coupon_id, true);

    if (!empty($affiliate->ID)) {
      Cookie::override($affiliate->ID);
      return true;
    }

    return false;
  }

  public function prefer_cookie_filter($use_referrer, $transaction, $customer) {
      return false;
  }

  protected function maybe_override_referrer_cookie($user_id) {
    $customer = new User($user_id);

    if ($customer->ID > 0) {
      $referrer = (int) $customer->referrer;

      if ($referrer > 0) {
        Cookie::override($referrer);
      }
    }
  }

  public function track_payment_rebill($orderData) {
    if (is_super_admin()) {
      return;
    }

    $this->log('Recording recurring payment');
    $this->log($orderData);
    $user_id = isset($orderData['member_id']) ? $orderData['member_id'] : '';
    $coupon = json_decode(isset($orderData['order_coupons']) ? $orderData['order_coupons'] : '');
    $coupon = isset($coupon[0]) ? $coupon[0] : '';
    $product = isset($orderData['order_products']) ? json_decode($orderData['order_products'], true)[0] : '';

    if (isset($coupon->id)) {
      if ($this->maybe_override_customer_referrer_with_coupon_affiliate($coupon->id)) {
          add_filter('esaf_prefer_referrer_over_cookie', [$this, 'prefer_cookie_filter'], 1, 3);
      }
    }

    $orderItem = new \MM_OrderItem($orderData['order_item_id']);

    if ($orderItem->isRecurring()) {
      $subNum = 'membermouse-sub-' . $orderItem->getId();
      $payNum = (int)\MM_TransactionLog::getTotalRecurringTransactionCountForItem($orderItem->getId());
      $payNum++;
    } else {
      $subNum = '';
      $payNum = 0;
    }

    $this->log('tracking new sale');
    Track::sale(
      'membermouse', /* source */
      $orderData['order_subtotal'] - $orderData['order_discount'], /* sale_amount */
      $orderData['order_transaction_id'], /* trans_num */
      $orderData['order_item_id'], /* item_id */
      $product['name'], /* item_name */
      $orderData['order_id'], /* order_id */
      isset($coupon->id) ? $coupon->id : 0, /* coupon */
      (int) $user_id, /* user_id */
      $subNum,
      $payNum
    );

    remove_filter('esaf_prefer_referrer_over_cookie', [$this, 'prefer_cookie_filter'], 1);
  }

  public function track_payment_received($orderData) {
    if (is_super_admin()) {
        return;
    }

    $this->log('Recording new payment');
    $this->log($orderData);
    $product = isset($orderData['order_products']) ? json_decode($orderData['order_products'], true)[0] : '';
    $user_id = isset($orderData['member_id']) ? $orderData['member_id'] : '';
    $coupon = json_decode(isset($orderData['order_coupons']) ? $orderData['order_coupons'] : '');
    $coupon = isset($coupon[0]) ? $coupon[0] : '';

    if (isset($coupon->id)) {
      if ($this->maybe_override_customer_referrer_with_coupon_affiliate($coupon->id)) {
        add_filter('esaf_prefer_referrer_over_cookie', [$this, 'prefer_cookie_filter'], 1, 3);
      }
    } else {
      $this->maybe_override_referrer_cookie($user_id);
    }

    $orderItem = new \MM_OrderItem($orderData['order_item_id']);

    if ($orderItem->isRecurring()) {
      $subNum = 'membermouse-sub-' . $orderItem->getId();
      $payNum = 1;
    } else {
      $subNum = '';
      $payNum = 0;
    }

    $this->log('tracking new sale');
    Track::sale(
      'membermouse', /* source */
      $orderData['order_subtotal'] - $orderData['order_discount'], /* sale_amount */
      $orderData['order_transaction_id'], /* trans_num */
      $orderData['order_item_id'], /* item_id */
      $product['name'], /* item_name */
      $orderData['order_id'], /* order_id */
      isset($coupon->id) ? $coupon->id : 0, /* coupon */
      (int) $user_id, /* user_id */
      $subNum,
      $payNum
    );
    remove_filter('esaf_prefer_referrer_over_cookie', [$this, 'prefer_cookie_filter'], 1);
  }

  /**
   * @return bool
   */
  public static function is_plugin_active() {
    return class_exists('MemberMouse');
  }

  public function track_refund($orderData) {
    $transactions = Transaction::get_all('', '', ['order_id' => $orderData['order_id'], 'source' => 'membermouse']);
    foreach ($transactions as $transaction) {
      $transaction->apply_refund($transaction->sale_amount);
      $transaction->store();
    }
  }

} //End class
