<?php

namespace EasyAffiliate\Controllers;

use EasyAffiliate\Lib\BaseCtrl;
use EasyAffiliate\Lib\Db;
use EasyAffiliate\Models\Campaign;
use EasyAffiliate\Models\Event;
use EasyAffiliate\Models\Creative;
use EasyAffiliate\Models\User;

/** This will pick up some of the hook based events, more in-depth
  * processing of certain events and event cleanup maintenance tasks.
  */
class EventsCtrl extends BaseCtrl {
  public function load_hooks() {
    // User based events
    add_action('wp_login', [$this, 'affiliate_login'], 10, 2);
    add_action('profile_update', [$this, 'affiliate_updated']);
    add_action('delete_user', [$this, 'affiliate_deleted']);

    // CPT based events
    add_action('wp_insert_post', [$this, 'post_added'], 10, 3);
    add_action('post_updated', [$this, 'post_updated']);
    add_action('delete_post', [$this, 'post_deleted']);

    // Ctax based events
    add_action('created_term', [$this, 'term_added'], 10, 3);
    add_action('edited_term', [$this, 'term_updated'], 10, 3);
    add_action('pre_delete_term', [$this, 'term_deleted'], 10, 2);

    // Affiliate application events
    add_action('esaf_cpt_create-esaf-application', [$this, 'application_submitted']);
    add_action('esaf_cpt_store_meta-esaf-application-status', [$this, 'application_status_changed'], 10, 3);

    // User events
    add_action('esaf_user_store_meta-is_affiliate', [$this, 'user_affiliate_status_changed'], 10, 3);
    add_action('esaf_user_store_meta-is_blocked', [$this, 'user_affiliate_blocked_changed'], 10, 3);

    // Transaction/Commission based events
    add_action('esaf_transaction_pre_update', [$this, 'transaction_changed'], 10, 2);
    add_action('esaf_commission_pre_update', [$this, 'commission_changed'], 10, 2);

    // Active/inactive affiliate events
    add_action('esaf_click_created', [$this, 'affiliate_active']);
    add_action('esaf_affiliate_inactive_check', [$this, 'affiliate_inactive']);

    if(!wp_next_scheduled('esaf_affiliate_inactive_check')) {
      wp_schedule_event(time() + DAY_IN_SECONDS, 'daily', 'esaf_affiliate_inactive_check');
    }
  }

  public function affiliate_login($user_login, $user) {
    $user = new User($user->ID);

    if($user->is_affiliate) {
      Event::record('affiliate-login', $user);
    }
  }

  public function affiliate_updated($affiliate_id) {
    if(!empty($affiliate_id)) {
      $affiliate = new User($affiliate_id);
      if($affiliate->is_affiliate) {
        Event::record('affiliate-updated', $affiliate);
      }
    }
  }

  public function affiliate_deleted($affiliate_id) {
    if(!empty($affiliate_id)) {
      // Since the 'delete_user' action fires just before the affiliate is deleted
      // we should still have access to the full User object for them
      $affiliate = new User($affiliate_id);
      if($affiliate->is_affiliate) {
        Event::record('affiliate-deleted', $affiliate);
      }
    }
  }

  public function post_added($post_id, $post, $update) {
    $post = get_post($post_id);
    if($post->post_type==Creative::$cpt && !$update) {
      Event::record('creative-added', (new Creative($post_id)));
    }
  }

  public function post_updated($post_id) {
    $post = get_post($post_id);
    if($post->post_type==Creative::$cpt) {
      Event::record('creative-updated', (new Creative($post_id)));
    }
  }

  public function post_deleted($post_id) {
    $post = get_post($post_id);
    if($post->post_type==Creative::$cpt) {
      Event::record('creative-deleted', (new Creative($post_id)));
    }
  }

  public function term_added($term_id, $tt_id, $taxonomy) {
    if($taxonomy==Campaign::$ctax) {
      Event::record('campaign-added', (new Campaign($term_id)));
    }
  }

  public function term_updated($term_id, $tt_id, $taxonomy) {
    if($taxonomy==Campaign::$ctax) {
      Event::record('campaign-updated', (new Campaign($term_id)));
    }
  }

  public function term_deleted($term_id, $taxonomy) {
    if($taxonomy==Campaign::$ctax) {
      Event::record('campaign-deleted', (new Campaign($term_id)));
    }
  }

  public function application_submitted($app) {
    Event::record('affiliate-application-submitted', $app);
  }

  public function application_status_changed($status, $old_status, $app) {
    if($status!=$old_status) {
      if( $status=='approved' &&
          ( 0 >= Event::get_count_by_obj(
              'affiliate-application-approved',
              'affiliate_application',
              $app->ID
            ) ) ) {
        Event::record('affiliate-application-approved', $app);
      }
      else if( $status=='ignored' &&
               ( 0 >= Event::get_count_by_obj(
                   'affiliate-application-ignored',
                   'affiliate_application',
                   $app->ID
                 ) ) ) {
        Event::record('affiliate-application-ignored', $app);
      }
    }
  }

  public function user_affiliate_status_changed($is_affiliate, $old_is_affiliate, $user) {
    if($is_affiliate!=$old_is_affiliate && $is_affiliate) {
      Event::record('affiliate-added', $user);
    } else if( ! $is_affiliate && $old_is_affiliate ) {
      Event::record('affiliate-deleted', $user);
    }
  }

  public function user_affiliate_blocked_changed($is_blocked, $old_is_blocked, $user) {
    if($is_blocked!=$old_is_blocked && $is_blocked) {
      Event::record('affiliate-blocked', $user);
    }
  }

  public function transaction_changed($transaction, $old_transaction) {
    if($old_transaction->refund_amount != $transaction->refund_amount && $transaction->refund_amount > 0.00) {
      Event::record('transaction-refunded', $transaction, ['refund_amount'=>$transaction->refund_amount]);
    }
  }

  public function commission_changed($commission, $old_commission) {
    if($old_commission->correction_amount != $commission->correction_amount && $commission->correction_amount > 0.00) {
      Event::record('commission-corrected', $commission, ['correction_amount'=>$commission->correction_amount]);
    }
  }

  public function affiliate_active($click) {
    if(empty($click->affiliate_id)) {
      return;
    }

    global $wpdb;
    $db = Db::fetch();

    $query = "SELECT
                COUNT(*) AS clicks,
                (
                  SELECT ev.id
                  FROM {$db->events} AS ev
                  WHERE ev.evt_id = %d
                  AND ev.evt_id_type = 'user'
                  AND ev.event = 'affiliate-active'
                  AND ev.created_at BETWEEN DATE_SUB(NOW(), INTERVAL 30 DAY) AND NOW()
                  LIMIT 1
                ) AS evt_id
              FROM {$db->clicks} cl
              WHERE cl.affiliate_id = %d
              AND cl.created_at BETWEEN DATE_SUB(NOW(), INTERVAL 30 DAY) AND NOW()
              AND cl.id <> %d";

    $result = $wpdb->get_row($wpdb->prepare($query, $click->affiliate_id, $click->affiliate_id, $click->id));

    if($result->clicks === '0' && $result->evt_id === null) {
      Event::record('affiliate-active', new User($click->affiliate_id));
    }
  }

  public function affiliate_inactive() {
    global $wpdb;
    $db = Db::fetch();

    $query = "SELECT DISTINCT cl.affiliate_id
              FROM $db->clicks cl
              LEFT JOIN (
                SELECT affiliate_id
                FROM $db->clicks
                WHERE created_at BETWEEN DATE_SUB(NOW(), INTERVAL 30 DAY) AND NOW()
                GROUP BY affiliate_id
              ) AS recent_clicks ON cl.affiliate_id = recent_clicks.affiliate_id
              LEFT JOIN (
                SELECT affiliate_id
                FROM $db->clicks
                WHERE created_at BETWEEN DATE_SUB(DATE_SUB(NOW(), INTERVAL 31 DAY), INTERVAL 1 SECOND)
                                     AND DATE_SUB(DATE_SUB(NOW(), INTERVAL 30 DAY), INTERVAL 1 SECOND)
                GROUP BY affiliate_id
              ) AS older_clicks ON cl.affiliate_id = older_clicks.affiliate_id
              LEFT JOIN (
                SELECT ev.evt_id
                FROM $db->events AS ev
                WHERE ev.evt_id_type = 'user'
                AND ev.event = 'affiliate-inactive'
                AND ev.created_at BETWEEN DATE_SUB(NOW(), INTERVAL 30 DAY) AND NOW()
                GROUP BY ev.evt_id
              ) AS inactive_affiliates ON cl.affiliate_id = inactive_affiliates.evt_id
              WHERE recent_clicks.affiliate_id IS NULL
              AND older_clicks.affiliate_id IS NOT NULL
              AND inactive_affiliates.evt_id IS NULL;";

    $result = $wpdb->get_col($query);

    foreach($result as $affiliate_id) {
      Event::record('affiliate-inactive', new User($affiliate_id));
    }
  }
} //End class
